#!/bin/bash
set -euo pipefail

# Color codes for output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# Helper functions
print_success() {
    echo -e "${GREEN}✓${NC} $1"
}

print_info() {
    echo -e "${BLUE}ℹ${NC} $1"
}

print_warning() {
    echo -e "${YELLOW}⚠${NC} $1"
}

print_error() {
    echo -e "${RED}✗${NC} $1"
}

# Check if running as root
if [ "$EUID" -ne 0 ]; then
    print_error "This script must be run as root"
    exit 1
fi

print_info "Starting Kopia installation and setup..."
echo

# Set timezone to Europe/Paris
print_info "Setting timezone to Europe/Paris..."
timedatectl set-timezone Europe/Paris
print_success "Timezone set to Europe/Paris"
echo

# Install required dependencies
print_info "Installing required dependencies (curl, gnupg)..."
apt-get update -qq
apt-get install -y -qq curl gnupg > /dev/null 2>&1
print_success "Dependencies installed"
echo

# Add Kopia GPG signing key
print_info "Adding Kopia GPG signing key..."
curl -s https://kopia.io/signing-key | gpg --batch --yes --dearmor -o /etc/apt/keyrings/kopia-keyring.gpg
print_success "GPG key added to /etc/apt/keyrings/kopia-keyring.gpg"
echo

# Add Kopia repository
print_info "Adding Kopia repository..."
echo "deb [signed-by=/etc/apt/keyrings/kopia-keyring.gpg] http://packages.kopia.io/apt/ stable main" | tee /etc/apt/sources.list.d/kopia.list > /dev/null
print_success "Repository added to /etc/apt/sources.list.d/kopia.list"
echo

# Update package lists and install Kopia
print_info "Updating package lists..."
apt-get update -qq
print_success "Package lists updated"
echo

print_info "Installing Kopia..."
apt-get install -y kopia
print_success "Kopia installed successfully"
echo

# Create systemd service file
print_info "Creating systemd service file..."
cat > /etc/systemd/system/kopia.service << 'EOF'
[Unit]
Description=Kopia Backup Server
Documentation=https://kopia.io/
After=network-online.target
Wants=network-online.target

[Service]
Type=simple
User=root
Group=root
Environment=HOME=/root
ExecStart=/usr/bin/kopia server start --insecure --address=0.0.0.0:51515 --without-password
Restart=on-failure
RestartSec=10s
StandardOutput=journal
StandardError=journal

[Install]
WantedBy=multi-user.target
EOF
print_success "Systemd service file created at /etc/systemd/system/kopia.service"
echo

# Reload systemd daemon
print_info "Reloading systemd daemon..."
systemctl daemon-reload
print_success "Systemd daemon reloaded"
echo

# Enable the Kopia service
print_info "Enabling Kopia service..."
systemctl enable kopia
print_success "Kopia service enabled (will start on boot)"
echo

# Start the Kopia service
print_info "Starting Kopia service..."
systemctl start kopia
print_success "Kopia service started"
echo

# Check service status
sleep 2
if systemctl is-active --quiet kopia; then
    print_success "Kopia service is running"
else
    print_warning "Kopia service may not be running properly"
    print_info "Check status with: systemctl status kopia"
fi
echo

# Display post-installation instructions
echo "=========================================="
echo -e "${GREEN}Kopia Installation Complete!${NC}"
echo "=========================================="
echo
print_info "Next steps:"
echo "  1. Access the Kopia web UI at: http://$(hostname -I | awk '{print $1}'):51515"
echo "  2. Connect to an existing repository OR create a new one via the web interface"
echo "  3. Configure your backup policies through the web UI"
echo
print_info "Service management commands:"
echo "  • Status:  systemctl status kopia"
echo "  • Stop:    systemctl stop kopia"
echo "  • Start:   systemctl start kopia"
echo "  • Restart: systemctl restart kopia"
echo "  • Logs:    journalctl -u kopia -f"
echo
print_warning "Security Notice:"
echo "  The service is currently running WITHOUT password authentication and WITHOUT TLS."
echo "  This is suitable for initial setup and testing."
echo "  For production use, consider:"
echo "    • Removing --without-password and configuring authentication"
echo "    • Removing --insecure and setting up TLS certificates"
echo "    • Restricting --address to specific network interface"
echo
echo "For more information, see the README.md file or visit https://kopia.io/docs/"
