#!/bin/bash
#
# Umami Analytics Installation Script for Debian 13 LXC
#
# This script installs Umami Analytics with:
# - PostgreSQL database
# - Bun runtime
# - systemd service management
#
# Usage: sudo ./install-umami.sh
#

set -euo pipefail

# ============================================
# Configuration
# ============================================

readonly INSTALL_DIR="/opt/umami"
readonly UMAMI_USER="umami"
readonly UMAMI_SERVICE="umami"
readonly DB_NAME="umami"
readonly DB_USER="umami"
readonly UMAMI_PORT="3000"
readonly UMAMI_VERSION="master"

# Color codes for output
readonly RED='\033[0;31m'
readonly GREEN='\033[0;32m'
readonly YELLOW='\033[1;33m'
readonly BLUE='\033[0;34m'
readonly NC='\033[0m'

# ============================================
# Helper Functions
# ============================================

log_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

command_exists() {
    command -v "$1" >/dev/null 2>&1
}

generate_password() {
    openssl rand -base64 32 | tr -d "=+/" | cut -c1-32
}

generate_app_secret() {
    openssl rand -base64 64 | tr -d "=+/" | cut -c1-64
}

check_root() {
    if [[ $EUID -ne 0 ]]; then
        log_error "This script must be run as root"
        exit 1
    fi
}

check_existing_installation() {
    if [[ -d "$INSTALL_DIR" ]]; then
        log_warning "Existing installation detected at $INSTALL_DIR"
        read -p "Do you want to remove it and reinstall? (yes/no): " -r
        if [[ $REPLY =~ ^[Yy][Ee][Ss]$ ]]; then
            log_info "Stopping existing service..."
            systemctl stop "$UMAMI_SERVICE" 2>/dev/null || true
            systemctl disable "$UMAMI_SERVICE" 2>/dev/null || true
            log_info "Removing existing installation..."
            rm -rf "$INSTALL_DIR"
            rm -f "/etc/systemd/system/${UMAMI_SERVICE}.service"
            systemctl daemon-reload
        else
            log_error "Installation cancelled"
            exit 1
        fi
    fi
}

# ============================================
# Installation Functions
# ============================================

install_prerequisites() {
    log_info "Updating package lists..."
    apt-get update -qq

    log_info "Installing system prerequisites..."
    apt-get install -y -qq \
        curl \
        wget \
        git \
        ca-certificates \
        gnupg \
        lsb-release \
        openssl \
        unzip \
        build-essential \
        postgresql \
        postgresql-contrib

    log_success "Prerequisites installed"
}

install_postgresql() {
    log_info "Starting PostgreSQL service..."
    systemctl start postgresql
    systemctl enable postgresql

    sleep 3

    log_success "PostgreSQL service started"
}

configure_postgresql() {
    log_info "Configuring PostgreSQL database..."

    local db_password
    db_password=$(generate_password)

    echo "$db_password" > /root/.umami_db_password
    chmod 600 /root/.umami_db_password

    sudo -u postgres psql -c "CREATE USER ${DB_USER} WITH PASSWORD '${db_password}';" 2>/dev/null || {
        log_warning "User ${DB_USER} already exists, updating password..."
        sudo -u postgres psql -c "ALTER USER ${DB_USER} WITH PASSWORD '${db_password}';"
    }

    sudo -u postgres psql -c "CREATE DATABASE ${DB_NAME} OWNER ${DB_USER};" 2>/dev/null || {
        log_warning "Database ${DB_NAME} already exists"
    }

    sudo -u postgres psql -c "GRANT ALL PRIVILEGES ON DATABASE ${DB_NAME} TO ${DB_USER};"

    sudo -u postgres psql -d "${DB_NAME}" -c "CREATE EXTENSION IF NOT EXISTS pgcrypto;"

    PGPASSWORD="${db_password}" psql -h localhost -U "${DB_USER}" -d "${DB_NAME}" -c "SELECT version();" >/dev/null

    if [[ $? -eq 0 ]]; then
        log_success "PostgreSQL configured successfully"
        export DB_PASSWORD="$db_password"
    else
        log_error "Failed to connect to PostgreSQL database"
        exit 1
    fi
}

create_umami_user() {
    log_info "Creating system user: ${UMAMI_USER}..."

    if id "$UMAMI_USER" &>/dev/null; then
        log_warning "User ${UMAMI_USER} already exists"
    else
        useradd --system --user-group --home-dir "$INSTALL_DIR" \
            --shell /usr/sbin/nologin --comment "Umami Analytics" "$UMAMI_USER"
        log_success "User ${UMAMI_USER} created"
    fi
}

install_bun() {
    log_info "Installing Bun runtime..."

    if command_exists bun; then
        log_warning "Bun already installed: $(bun --version)"
        return
    fi

    curl -fsSL https://bun.sh/install | bash

    export BUN_INSTALL="/root/.bun"
    export PATH="$BUN_INSTALL/bin:$PATH"

    # Copy Bun binary to system location (not symlink, so umami user can access it)
    cp "$BUN_INSTALL/bin/bun" /usr/local/bin/bun
    chmod 755 /usr/local/bin/bun

    if command_exists bun; then
        log_success "Bun installed: $(bun --version)"
    else
        log_error "Failed to install Bun"
        exit 1
    fi
}

clone_umami() {
    log_info "Cloning Umami repository to ${INSTALL_DIR}..."

    git clone --depth 1 --branch "$UMAMI_VERSION" \
        https://github.com/umami-software/umami.git "$INSTALL_DIR"

    log_success "Repository cloned"
}

install_dependencies() {
    log_info "Installing Umami dependencies with Bun (this may take a few minutes)..."

    cd "$INSTALL_DIR"

    export BUN_INSTALL_TIMEOUT=300000

    bun install --frozen-lockfile 2>&1 | tee /tmp/umami-install.log

    if [[ ${PIPESTATUS[0]} -eq 0 ]]; then
        log_success "Dependencies installed successfully"
    else
        log_error "Failed to install dependencies. Check /tmp/umami-install.log"
        exit 1
    fi
}

create_env_file() {
    log_info "Creating environment configuration..."

    local app_secret
    app_secret=$(generate_app_secret)

    local database_url
    database_url="postgresql://${DB_USER}:${DB_PASSWORD}@localhost:5432/${DB_NAME}"

    cat > "${INSTALL_DIR}/.env" <<EOF
# Database Configuration
DATABASE_URL=${database_url}

# Application Secret (for JWT signing and encryption)
APP_SECRET=${app_secret}

# Server Configuration
PORT=${UMAMI_PORT}
HOSTNAME=0.0.0.0

# Disable telemetry
NEXT_TELEMETRY_DISABLED=1

# Tracker Configuration
TRACKER_SCRIPT_NAME=umamii.js

# Environment
NODE_ENV=production
EOF

    chmod 600 "${INSTALL_DIR}/.env"

    log_success "Environment file created at ${INSTALL_DIR}/.env"

    local server_ip
    server_ip=$(hostname -I | awk '{print $1}')

    cat > /root/umami-credentials.txt <<EOF
=================================================
UMAMI ANALYTICS INSTALLATION CREDENTIALS
=================================================

Web Interface:
  URL: http://${server_ip}:${UMAMI_PORT}
  Username: admin
  Password: umami

  !!! IMPORTANT: Change the default password immediately after first login !!!

Database:
  Host: localhost
  Port: 5432
  Database: ${DB_NAME}
  Username: ${DB_USER}
  Password: ${DB_PASSWORD}

Configuration File: ${INSTALL_DIR}/.env
Service Name: ${UMAMI_SERVICE}

=================================================
Generated on: $(date)
=================================================
EOF

    chmod 600 /root/umami-credentials.txt
    log_info "Credentials saved to /root/umami-credentials.txt"
}

build_umami() {
    log_info "Building Umami application (this may take several minutes)..."

    cd "$INSTALL_DIR"

    bun run build 2>&1 | tee /tmp/umami-build.log

    if [[ ${PIPESTATUS[0]} -eq 0 ]]; then
        log_success "Umami built successfully"
    else
        log_error "Build failed. Check /tmp/umami-build.log for details"
        log_error "Common issues:"
        log_error "  - DATABASE_URL not set correctly"
        log_error "  - PostgreSQL connection issues"
        log_error "  - Network issues downloading GeoIP database"
        exit 1
    fi

    if [[ -d "${INSTALL_DIR}/.next" ]] && [[ -f "${INSTALL_DIR}/public/script.js" ]]; then
        log_success "Build artifacts verified"
    else
        log_error "Build completed but expected artifacts are missing"
        exit 1
    fi

    # Copy public directory to standalone build
    log_info "Copying static assets to standalone server..."
    cp -r "${INSTALL_DIR}/public" "${INSTALL_DIR}/.next/standalone/"

    # Copy .next/static directory to standalone build (required for Next.js standalone mode)
    cp -r "${INSTALL_DIR}/.next/static" "${INSTALL_DIR}/.next/standalone/.next/"

    if [[ -d "${INSTALL_DIR}/.next/standalone/public" ]] && [[ -d "${INSTALL_DIR}/.next/standalone/.next/static" ]]; then
        log_success "Static assets copied successfully"
    else
        log_error "Failed to copy static assets to standalone build"
        exit 1
    fi
}

set_permissions() {
    log_info "Setting file permissions..."

    chown -R "${UMAMI_USER}:${UMAMI_USER}" "$INSTALL_DIR"

    # Ensure standalone directory has correct permissions
    if [[ -d "${INSTALL_DIR}/.next/standalone" ]]; then
        chown -R "${UMAMI_USER}:${UMAMI_USER}" "${INSTALL_DIR}/.next/standalone"
        log_info "Standalone directory permissions set"
    fi

    chmod 600 "${INSTALL_DIR}/.env"

    find "$INSTALL_DIR" -type d -exec chmod 755 {} \;
    find "$INSTALL_DIR" -type f -exec chmod 644 {} \;

    if [[ -d "${INSTALL_DIR}/node_modules/.bin" ]]; then
        chmod -R 755 "${INSTALL_DIR}/node_modules/.bin"
    fi

    log_success "Permissions set correctly"
}

create_systemd_service() {
    log_info "Creating systemd service..."

    cat > "/etc/systemd/system/${UMAMI_SERVICE}.service" <<EOF
[Unit]
Description=Umami Analytics
Documentation=https://umami.is/docs
After=network.target postgresql.service
Requires=postgresql.service

[Service]
Type=simple
User=${UMAMI_USER}
Group=${UMAMI_USER}
WorkingDirectory=${INSTALL_DIR}
Environment="PATH=/usr/local/bin:/usr/bin:/bin"
EnvironmentFile=${INSTALL_DIR}/.env

# Start command using standalone server
ExecStart=/usr/local/bin/bun ${INSTALL_DIR}/.next/standalone/server.js

# Restart policy
Restart=always
RestartSec=10s

# Security hardening
NoNewPrivileges=true
PrivateTmp=true
ProtectSystem=strict
ProtectHome=true
ReadWritePaths=${INSTALL_DIR}

# Resource limits
LimitNOFILE=65536
TimeoutStartSec=120
TimeoutStopSec=30

# Logging
StandardOutput=journal
StandardError=journal
SyslogIdentifier=${UMAMI_SERVICE}

[Install]
WantedBy=multi-user.target
EOF

    systemctl daemon-reload

    log_success "Systemd service created"
}

start_service() {
    log_info "Enabling and starting ${UMAMI_SERVICE} service..."

    systemctl enable "$UMAMI_SERVICE"
    systemctl start "$UMAMI_SERVICE"

    sleep 5

    if systemctl is-active --quiet "$UMAMI_SERVICE"; then
        log_success "Umami service is running"
    else
        log_error "Umami service failed to start"
        log_error "Check logs with: journalctl -u ${UMAMI_SERVICE} -n 50"
        exit 1
    fi
}

verify_installation() {
    log_info "Verifying installation..."

    sleep 10

    local server_ip
    server_ip=$(hostname -I | awk '{print $1}')

    if ss -tln | grep -q ":${UMAMI_PORT}"; then
        log_success "Umami is listening on port ${UMAMI_PORT}"
    else
        log_warning "Port ${UMAMI_PORT} is not listening yet"
    fi

    if curl -sf "http://localhost:${UMAMI_PORT}/api/heartbeat" >/dev/null 2>&1; then
        log_success "Application health check passed"
    else
        log_warning "Health check failed - application may still be starting up"
        log_info "Check status with: journalctl -u ${UMAMI_SERVICE} -f"
    fi

    # Verify standalone build exists
    if [[ -f "${INSTALL_DIR}/.next/standalone/server.js" ]]; then
        log_success "Standalone server verified"
    else
        log_warning "Standalone server.js not found - this may cause issues"
    fi

    # Verify public assets in standalone
    if [[ -f "${INSTALL_DIR}/.next/standalone/public/script.js" ]]; then
        log_success "Tracker script verified in standalone build"
    else
        log_warning "Tracker script not found in standalone build"
    fi

    echo ""
    echo "=========================================="
    log_success "UMAMI INSTALLATION COMPLETED!"
    echo "=========================================="
    echo ""
    log_info "Access Umami at: http://${server_ip}:${UMAMI_PORT}"
    log_info "Default credentials:"
    log_info "  Username: admin"
    log_info "  Password: umami"
    echo ""
    log_warning "IMPORTANT: Change the default password immediately!"
    echo ""
    log_info "Useful commands:"
    log_info "  Service status: systemctl status ${UMAMI_SERVICE}"
    log_info "  View logs:      journalctl -u ${UMAMI_SERVICE} -f"
    log_info "  Restart:        systemctl restart ${UMAMI_SERVICE}"
    log_info "  Stop:           systemctl stop ${UMAMI_SERVICE}"
    echo ""
    log_info "Credentials saved: /root/umami-credentials.txt"
    log_info "PostgreSQL password: /root/.umami_db_password"
    echo ""
}

# ============================================
# Main Execution
# ============================================

main() {
    log_info "Starting Umami Analytics installation on Debian 13 LXC..."
    echo ""

    check_root
    check_existing_installation

    install_prerequisites
    install_postgresql
    configure_postgresql
    create_umami_user
    install_bun
    clone_umami
    install_dependencies
    create_env_file
    build_umami
    set_permissions
    create_systemd_service
    start_service
    verify_installation
}

trap 'log_error "Installation failed at line $LINENO"' ERR

main "$@"
